unit XFileCtr;

{
  ==========================
  eXtended File Controls 1.0 beta 20 (1999-09-24)
  ==========================

  This unit should be a COMPLETE REPLACEMENT of Borland FileCtrl unit.
  But you can still use both components at once because of new naming conventions.
  Five components works only with each other not with the standard ones.

  TXDriveComboBox	modified Jordan Russell's component
  TXDirectoryListBox	modified Jordan Russell's component
  TXFileListBox		core of the collection
			widely modified and extended Borland original
  TXFilterComboBox	the same as Borland's original
  TXPathLabel           modified Jordan Russell's component

  Target: Delphi 4.
  
  Jordan Russell is author of Inno Setup Compiler. He made some useful
  modifications of Borland standard components. I made some useful modifications
  of his components :-). Thank you.

  Freeware.

  Copyright  Roman Stedronsky 1998 - 1999, Roman.Stedronsky@seznam.cz

  All rights reserved. You may use this software in an application
  without fee or royalty, provided this copyright notice remains intact.

  TXFileListBox
  -------------
  An objective of my mission was to offer the same functionality like listbox
  used in Windows Commander (shareware file manager). Design is very familiar
  and keyboard shortcuts too. But speed... yea, speed is slower...

  methods

    FormatDirName		if name is an directory, add brackets (ex. '[WINDOWS]')
    UnFormatDirName		remove brackets (ex. 'WINDOWS')
    GetItemType			type of an item

  public properties

    FilesCount			amount of files in listbox
    FoldersCount		amount of folders in listbox
    Parametres			command line parametres for opening file
    FileNames[Index: integer]   file name with extension (use instead of Items[index]!)
    FullFileNames[Index: integer]	path + filename + extension
    Dates[Index: integer]	date of given file
    Sizes[Index: integer]	size of given file
    Attributes[Index: integer]	attribute of given file

  published properties

    DirLabel			TXPathLabel for showing current path
    DirLabelMask		show path with current mask
    FileType			files to be shown
    SortBy			sort by name, size, date, time or attribute
    SortOrder			ascending or descending
    SortFolders			how sort folders - like files or always by name
    DirName			string used for marking folders (ex. 'ADR')
    DateFormat			format for displaying dates (ex. 'yy-mm-dd')
    TimeFormat			format for displaying times (ex. 'yy-mm-dd')
    ShowItems			which parts of file info show (date, time, size, attr)
    ShellIcons			display true shell icons or generic icons (much faster)
    ShowParentFolder		show folder '..' for going to parent folder
    ShowFolderAnimation		show little animation, when opening folders
    WidthFileName		width of filename part
    WidthSize			width of size part
    WidthDate			width of date part
    WidthTime			width of time part
    WidthAttribute		width of attribute part
    OpenParentFolders		enable moving to the parent folder (Ctrl + PgUp)
    OpenFolders                 enable moving to child folders (Enter, Ctrl + PgDown)
    OpenFiles			enable open file (Enter)

  events

    OnOpenFile			event on opening file
    OnOpenFolder		event on opening folder
    OnUpdateContent		event on updating listbox (rereading files and folders)
    OnAcceptFile		event for custom file / folder filtering
    				if you accept file, set Result := true, otherwise false
                                Define NO_CUSTOM_FILTER conditional for disabling this feature.

  possible future improvements:

   - tooltip info for long filename
   - improved speed (espacielly drawing)
   - custom file list reading (redefine FindFirst, FindNext)?
   - smart icon caching: cache all, cache folder only
   - delayed true shell icon displaying
   - select: files & folders, only files
   - quick search - Alt + Character
   - file management - Copy, Move, Delete (full, to Recycled Bin)
   - 2 file names modes - long, short (DOS)
   - folders with sizes
   - shell context menu and properties
   - AutoColumnWidth - auto calculate width of an column
}

interface

{ $DEFINE NO_CUSTOM_FILTER}

uses
  Windows, StdCtrls, Classes, Graphics, Controls, Messages, SysUtils,
  XSortLst { listbox with general sorting functionality },
  XFileUtl { functions for working with files and their names };

type
  { TXFileListBox types}
  TIconArray = array[Abs(-3 {iiDefault})..Abs(-16 {iiShortcut})] of TIcon; // array of default icons TFileListBox
  TFileItem = (fiFileName, fiSize, fiDate, fiTime, fiAttribute);
  TShowItems = set of TFileItem;
  TItemType = (itParentFolder, itFolder, itFile, itUnknown); // general info for a list item
  TSortOrder = (soAscending, soDescending); // sort order type
  TSortFolders = (sfAlwaysByName, sfLikeFiles); // sort folders like files ?
  TCompareArray = array[TFileItem] of TSortStringListCompare; // array of compare functions
  TXFileAttr = (xftNormal, xftDirectory, xftReadOnly, xftArchive, xftHidden,
    xftSystem, xftCompressed, xftOffline, xftTemporary, xftVolumeID); // file attributes
  TXFileType = set of TXFileAttr; // set of file attributes
  TOpenFileEvent = procedure(FileName: string) of object; // for custom handling of opening files
  TOpenFolderEvent = procedure(Folder: string) of object; // for custom handling of opening folders
  TUpdateContentEvent = procedure of object; // event type
{$IFNDEF NO_CUSTOM_FILTER}
  TAcceptFileEvent = function(const ItemType: TItemType; const FileInfo: TSearchRec) {; var Accept}: boolean of object;
{$ENDIF}
  PData = ^TData;
  TData = record // basic data structure for each file
    IconIndex: integer;
    Date: TDateTime;
    Size: integer;
    Attribute: integer;
    FileNameLength: integer;
  end;

  { TXDriveComboBox types }
  TXDriveType = (dtUnknown, dtNoDrive, dtFloppy, dtFixed, dtNetwork, dtCDROM, dtRAM); // types of drives
  TXDriveIcons = (dcFloppyBMP, dcFixedBMP, dcNetworkBMP, dcCDROMBMP, dcRAMBMP); // bitmaps for drives
  TXTextCase = (tcLowerCase, tcUpperCase); // case of drive name

  { TXDirectoryListBox types }
  TXDirectoryIcons = (dcClosedBMP, dcOpenedBMP, dcCurrentBMP); // bitmaps for directories

const
  { TXFileListBox constants }

  { generic icon types TIconInfo }
  iiUnknown = -2;
  iiDefault = -3;
  iiDefaultHidden = -4;
  iiFolder = -5;
  iiFolderOpen = -6;
  iiFolderParent = -7;
  iiFolderShared = -8;
  iiFolderHidden = -9;
  iiFolderSystem = -10;
  iiFolderHiddenSystem = -11;
  iiExeDos = -12;
  iiExeWin = -13;
  iiCompress = -14;
  iiCompressOpen = -15;
  iiShortcut = -16;

  { default values of some properties }
  cWidthFileName = 150;
  cWidthSize = 60;
  cWidthDate = 45;
  cWidthTime = 30;
  cWidthAttribute = 30;
  cDefaultDateFormat = 'dd.MM.yy';
  cDefaultTimeFormat = 'hh:mm';
  cDefaultDirName = '<DIR>';
  cDefaultShowItems = [fiFileName, fiSize, fiDate, fiTime, fiAttribute];
  cDefaultFileType = [xftArchive, xftCompressed, xftDirectory, xftHidden, xftNormal, xftReadOnly, xftSystem];

type
  TXDriveComboBox = class;
  TXDirectoryListBox = class;
  TXFileListBox = class;
  TXFilterComboBox = class;
  TXPathLabel = class;

  { TXDriveComboBox }

  TXDriveComboBox = class(TCustomComboBox)
  private
    FDirList: TXDirectoryListBox;
    FDrive: Char;
    FTextCase: TXTextCase;
    procedure SetDirListBox(Value: TXDirectoryListBox);
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
    procedure CMColorChanged(var Message: TMessage); message CM_COLORCHANGED;
    procedure CMSysColorChange(var Message: TMessage); message CM_SYSCOLORCHANGE;
    procedure SetDrive(NewDrive: Char);
    procedure SetTextCase(XTextCase: TXTextCase);
    procedure ReadIcons;
    procedure FreeIcons;
    procedure ResetItemHeight;
  protected
    Icons: array[TXDriveIcons] of TIcon;
    procedure CreateWnd; override;
    procedure DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState); override;
    procedure Click; override;
    procedure BuildList; virtual;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Text;
    property Drive: Char read FDrive write SetDrive;
  published
    property Anchors;
    property Color;
    property Constraints;
    property Ctl3D;
    property DirList: TXDirectoryListBox read FDirList write SetDirListBox;
    property DragMode;
    property DragCursor;
    property Enabled;
    property Font;
    property ImeMode;
    property ImeName;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property TextCase: TXTextCase read FTextCase write SetTextCase default tcLowerCase;
    property Visible;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnDropDown;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnStartDrag;
  end;

  { TXDirectoryListBox }

  TXDirectoryListBox = class(TCustomListBox)
  private
    FFileList: TXFileListBox;
    FDriveCombo: TXDriveComboBox;
    FDirLabel: TXPathLabel;
    FInSetDir: Boolean;
    FPreserveCase: Boolean;
    FCaseSensitive: Boolean;
    function GetDrive: Char;
    procedure SetFileListBox(Value: TXFileListBox);
    procedure SetDirLabel(Value: TXPathLabel);
    procedure SetDirLabelCaption;
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
    procedure CMColorChanged(var Message: TMessage); message CM_COLORCHANGED;
    procedure CMSysColorChange(var Message: TMessage); message CM_SYSCOLORCHANGE;
    procedure SetDrive(Value: Char);
    procedure DriveChange(NewDrive: Char);
    procedure SetDir(const NewDirectory: string);
    procedure SetDirectory(const NewDirectory: string); virtual;
    procedure ResetItemHeight;
    procedure ReadIcons;
    procedure FreeIcons;
  protected
    Icons: array[TXDirectoryIcons] of TIcon;
    FDirectory: string;
    FOnChange: TNotifyEvent;
    procedure Change; virtual;
    procedure DblClick; override;
    procedure CreateWnd; override;
    procedure DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState); override;
    function ReadDirectoryNames(const ParentDirectory: string; DirectoryList: TStringList): Integer;
    procedure BuildList; virtual;
    procedure KeyPress(var Key: Char); override;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function DisplayCase(const S: string): string;
    function FileCompareText(const A, B: string): Integer;
    function GetItemPath(Index: Integer): string;
    procedure OpenCurrent;
    procedure Refresh;
    property Drive: Char read GetDrive write SetDrive;
    property Directory: string read FDirectory write SetDirectory;
    property PreserveCase: Boolean read FPreserveCase;
    property CaseSensitive: Boolean read FCaseSensitive;
  published
    property Align;
    property Anchors;
    property Color;
    property Constraints;
    property Columns;
    property Ctl3D;
    property DirLabel: TXPathLabel read FDirLabel write SetDirLabel;
    property DragCursor;
    property DragMode;
    property Enabled;
    property FileList: TXFileListBox read FFileList write SetFileListBox;
    property Font;
    property ImeMode;
    property ImeName;
    property IntegralHeight;
    property ItemHeight;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDrag;
  end;

  { TXFileListBox }

  TXFileListBox = class(TCustomListBox)
  { original methods and properties }
  private
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
    function GetDrive: char;
    function IsMaskStored: Boolean;
    procedure SetDrive(Value: char);
    procedure SetFileEdit(Value: TEdit);
    function SetDirectory(const NewDirectory: string): boolean;
    procedure SetMask(const NewMask: string);
    procedure SetFileName(const NewFile: string);
    procedure ResetItemHeight;
  protected
    FDirectory: string;
    FMask: string;
    FFileEdit: TEdit;
    FDirList: TXDirectoryListBox;
    FFilterCombo: TXFilterComboBox;
    FOnChange: TNotifyEvent;
    FLastSel: Integer;
    FShowGlyphs: Boolean;
    procedure CreateWnd; override;
    procedure Click; override;
    procedure Change; virtual;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  published
    property Align;
    property Anchors;
    property Color;
    property Constraints;
    property Ctl3D;
    property DragCursor;
    property DragMode;
    property Enabled;
    property ExtendedSelect;
    property FileEdit: TEdit read FFileEdit write SetFileEdit;
    property Font;
    property ImeMode;
    property ImeName;
    property IntegralHeight;
    property ItemHeight;
    property Mask: string read FMask write SetMask stored IsMaskStored;
    property MultiSelect;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDrag;
  { redefined methods and properties }
  protected
    procedure ReadFileNames; virtual;
    procedure DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState); override;
    procedure DblClick; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure SetShowGlyphs(Value: Boolean);
    function GetFilePath: string; virtual;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure ApplyFilePath(const EditText: string); virtual;
    procedure Update; reintroduce;
    property Drive: char read GetDrive write SetDrive;
    property FileName: string read GetFilePath write ApplyFilePath;
    property Directory: string read FDirectory write ApplyFilePath; // when not published, autosize = true causes hiding of the component
  published
    property ShowGlyphs: Boolean read FShowGlyphs write SetShowGlyphs default true;
  { new methods and properties }
  protected
    { internal variables for properties }
    FAutoWidth: boolean;
    FCached: boolean;
    FFoldersCount: integer;
    FFilesCount: integer;
    FFileType: TXFileType;
    FDirLabel: TXPathLabel;
    FDirLabelMask: boolean;
    FShowItems: TShowItems;
    FShellIcons: boolean;
    FShowParentFolder: boolean;
    FShowFolderAnimation: boolean;
    FBeepErrors: boolean;
    FOnOpenFile: TOpenFileEvent;
    FOnOpenFolder: TOpenFolderEvent;
    FOnUpdateContent: TUpdateContentEvent;
{$IFNDEF NO_CUSTOM_FILTER}
    FOnAcceptFile: TAcceptFileEvent;
{$ENDIF}
    FWidthFileName: word;
    FWidthSize: word;
    FWidthDate: word;
    FWidthTime: word;
    FWidthAttribute: word;
    FParametres: string;
    FDateFormat: string;
    FTimeFormat: string;
    FDirName: string;
    FSortBy: TFileItem;
    FSortOrder: TSortOrder;
    FSortFolders: TSortFolders;
    FOpenFolders: boolean;
    FOpenParentFolders: boolean;
    FOpenFiles: boolean;
    { other variables }
    FSortList: TSortStringList;
    FIcons: TStringList;
    FStdIcons: TIconArray;
    FCompareFunctions: TCompareArray;
    FSortOrderMultiplier: integer;
    FParentFolder: integer;
    FFirstFolder: integer;
    FLastFolder: integer;
    { internal methods }
    function GenerateIcon(Index: integer): TIcon;
    function GetFileName(Index: integer): string; overload;
    function GetFileName: string; overload;
    procedure OpenParentFolder;
    procedure OpenFolder;
    procedure OpenFile;
    procedure LoadStdIcons;
    procedure FreeStdIcons;
    procedure FreeIcons;
    procedure FreeData;
    function GetData(StringList: TStrings; Index: integer): PData; overload;
    function GetData(Index: integer): PData; overload;
    function GetType(IconIndex: integer): TItemType;
    function GetItemType(StringList: TStrings; Index: integer): TItemType; overload; // general type function
    { sort related functions }
    procedure ReSort; // change sorting without rereading files
    procedure Sort; // sort function
    function CompareFileName(const Index1, Index2: integer): integer; // string compare function
    function CompareSize(const Index1, Index2: integer): integer;
    function CompareDate(const Index1, Index2: integer): integer;
    function CompareTime(const Index1, Index2: integer): integer;
    function CompareAttribute(const Index1, Index2: integer): integer;
    procedure AdjustWidth; virtual;
    { read and write methods for properties }
    procedure FWriteAutoWidth(Value: boolean);
    procedure FWriteDirLabel(Value: TXPathLabel);
    procedure SetDirLabelCaption;
    procedure FWriteDirLabelMask(Value: boolean);
    procedure FWriteFileType(Value: TXFileType);
    procedure FWriteSortBy(Value: TFileItem);
    procedure FWriteSortOrder(Value: TSortOrder);
    procedure FWriteSortFolders(Value: TSortFolders);
    procedure FWriteDirName(Value: string);
    procedure FWriteDateFormat(Value: string);
    procedure FWriteTimeFormat(Value: string);
    procedure FWriteShowItems(Value: TShowItems);
    procedure FWriteShellIcons(Value: boolean);
    procedure FWriteShowParentFolder(Value: boolean);
    procedure FWriteWidthFileName(Value: word);
    procedure FWriteWidthSize(Value: word);
    procedure FWriteWidthDate(Value: word);
    procedure FWriteWidthTime(Value: word);
    procedure FWriteWidthAttribute(Value: word);
    procedure FWriteCached(Value: boolean);
    function FReadFileNames(Index: integer): string;
    procedure FWriteFileNames(Index: integer; Value: string);
    function FReadFullFileNames(Index: integer): string;
    function FReadDate(Index: integer): TDateTime;
    function FReadSize(Index: integer): integer;
    function FReadAttribute(Index: integer): TXFileType;
  public
    procedure Open;
    function FormatDirName(Dir: string): string;
    function UnFormatDirName(Dir: string): string;
    function GetItemType(Index: integer): TItemType; overload; // type of an item in currently used Items
    function GetItemType: TItemType; overload; // type of an actual item
    property FilesCount: integer read FFilesCount;
    property FoldersCount: integer read FFoldersCount;
    property FileNames[Index: integer]: string read FReadFileNames write FWriteFileNames; default;
    property FullFileNames[Index: integer]: string read FReadFullFileNames;
    property Parametres: string read FParametres write FParametres;
    property Dates[Index: integer]: TDateTime read FReadDate;
    property Sizes[Index: integer]: integer read FReadSize;
    property Attributes[Index: integer]: TXFileType read FReadAttribute;
  published
    property AutoWidth: boolean read FAutoWidth write FWriteAutoWidth default true;
//    property Cached: boolean read FCached write FWriteCached;
    property DirLabel: TXPathLabel read FDirLabel write FWriteDirLabel;
    property DirLabelMask: boolean read FDirLabelMask write FWriteDirLabelMask;
    property FileType: TXFileType read FFileType write FWriteFileType default [xftNormal]; //redefined
    property SortBy: TFileItem read FSortBy write FWriteSortBy default fiFileName;
    property SortOrder: TSortOrder read FSortOrder write FWriteSortOrder default soAscending;
    property SortFolders: TSortFolders read FSortFolders write FWriteSortFolders default sfAlwaysByName;
    property DirName: string read FDirName write FWriteDirName;
    property DateFormat: string read FDateFormat write FWriteDateFormat;
    property TimeFormat: string read FTimeFormat write FWriteTimeFormat;
    property ShowItems: TShowItems read FShowItems write FWriteShowItems default cDefaultShowItems;
    property ShellIcons: boolean read FShellIcons write FWriteShellIcons default true;
    property ShowParentFolder: boolean read FShowParentFolder write FWriteShowParentFolder default true;
    property ShowFolderAnimation: boolean read FShowFolderAnimation write FShowFolderAnimation default true;
    property BeepErrors: boolean read FBeepErrors write FBeepErrors default true;
    property WidthFileName: word read FWidthFileName write FWriteWidthFileName default cWidthFileName;
    property WidthSize: word read FWidthSize write FWriteWidthSize default cWidthSize;
    property WidthDate: word read FWidthDate write FWriteWidthDate default cWidthDate;
    property WidthTime: word read FWidthTime write FWriteWidthTime default cWidthTime;
    property WidthAttribute: word read FWidthAttribute write FWriteWidthAttribute default cWidthAttribute;
    property OpenParentFolders: boolean read FOpenParentFolders write FOpenParentFolders default true;
    property OpenFolders: boolean read FOpenFolders write FOpenFolders default true;
    property OpenFiles: boolean read FOpenFiles write FOpenFiles default true;
    property OnOpenFile: TOpenFileEvent read FOnOpenFile write FOnOpenFile;
    property OnOpenFolder: TOpenFolderEvent read FOnOpenFolder write FOnOpenFolder;
    property OnUpdateContent: TUpdateContentEvent read FOnUpdateContent write FOnUpdateContent;
{$IFNDEF NO_CUSTOM_FILTER}
    property OnAcceptFile: TAcceptFileEvent read FOnAcceptFile write FOnAcceptFile;
{$ENDIF}
  end;

  { TXFilterComboBox }

  TXFilterComboBox = class(TCustomComboBox)
  private
    FFilter: string;
    FFileList: TXFileListBox;
    MaskList: TStringList;
    function IsFilterStored: Boolean;
    function GetMask: string;
    procedure SetFilter(const NewFilter: string);
    procedure SetFileListBox(Value: TXFileListBox);
  protected
    procedure Change; override;
    procedure CreateWnd; override;
    procedure Click; override;
    procedure BuildList;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Mask: string read GetMask;
    property Text;
  published
    property Anchors;
    property Color;
    property Constraints;
    property Ctl3D;
    property DragMode;
    property DragCursor;
    property Enabled;
    property FileList: TXFileListBox read FFileList write SetFileListBox;
    property Filter: string read FFilter write SetFilter stored IsFilterStored;
    property Font;
    property ImeName;
    property ImeMode;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnDropDown;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnStartDrag;
  end;

  { TNewPathLabel }

  TXPathLabel = class(TCustomLabel)
  protected
    procedure Paint; override;
  public
    constructor Create(AOwner: TComponent); override;
  published
    property Align;
    property Alignment;
    property Anchors;
    property AutoSize;
    property BiDiMode;
    property Caption stored False;
    property Color;
    property Constraints;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property FocusControl;
    property Font;
    property ParentBiDiMode;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowAccelChar;
    property ShowHint;
    property Transparent;
    property Layout;
    property Visible;
    property WordWrap;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDock;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDock;
    property OnStartDrag;
  end;

function MinimizeName(const Filename: string; Canvas: TCanvas; MaxLen: Integer): string;

procedure Register;

implementation

{$R XFileCtr.res}

uses
  Forms, ShellAPI;

resourcestring
  rsDefaultFilter = 'All files (*.*)|*.*';
  rsInvalidFileName = 'Invalid file name - %s';

const
  DefaultMask = '*.*';
  cAttributeStrings: array[TXFileAttr] of string = ('', 'd', 'r', 'a', 'h', 's', 'c', 'o', 't', 'v');
  cEmptyStr = '-'; //''
  cStdIcons: array[Abs(iiDefault)..Abs(iiShortcut)] of PChar = ('_DEFAULT',
    '_DEFAULTHIDDEN', '_FOLDER', '_FOLDEROPEN', '_FOLDERPARENT', '_FOLDERSHARED',
    '_FOLDERHIDDEN', '_FOLDERSYSTEM', '_FOLDERHIDDENSYSTEM', '_EXEDOS', '_EXEWIN', '_COMPRESS', '_COMPRESSOPEN', '_SHORTCUT');
  cOffsetWidth = 2;
  cSpaceWidth = 4;

  faNormal = $00000080;
  faTemporary = $00000100;
  faCompressed = $00000800;
  faOffline = $00001000;

  cAttributes: array[TXFileAttr] of integer = (faNormal, faDirectory, faReadOnly,
    faArchive, faHidden, faSysFile, faCompressed, faOffline, faTemporary, faVolumeID);

  DirListMinHeight = 16; //15;
  DriveComboMinHeight = 16;
  MaxResourceNameLength = 32;
  cDriveIcons: array[TXDriveIcons] of PChar = ('_FLOPPY', '_HARD', '_NETWORK', '_CDROM', '_RAM');
  cDirectoryIcons: array[TXDirectoryIcons] of PChar = ('_FOLDER', '_FOLDEROPEN', '_FOLDERCURRENT');
  NewResourceSuffix = '95';

{ ************ SUBSIDIARY FUNCTIONS ************ }

function AttrToStr(Value: integer): string;
var
  Counter: TXFileAttr;
begin
  Result := '';
  for Counter := Low(TXFileAttr) to High(TXFileAttr) do
    if not (Counter in [xftNormal, xftDirectory, xftOffline, xftTemporary, xftVolumeID]) then
      if Value and cAttributes[Counter] <> 0 then
        Result := Result + cAttributeStrings[Counter]
      else
        Result := Result + cEmptyStr;
end;

function AttrToFileType(Value: integer): TXFileType;
var
  Counter: TXFileAttr;
begin
  Result := [];
  for Counter := Low(TXFileAttr) to High(TXFileAttr) do
    if Value and cAttributes[Counter] <> 0 then
      Result := Result + [Counter];
end;

function FileTypeToAttr(Value: TXFileType): integer;
var
  Index: TXFileAttr;
begin
  Result := 0;
  for Index := Low(TXFileAttr) to High(TXFileAttr) do
    if Index in Value then
      Result := Result or cAttributes[Index];
end;

function GetItemHeight(Font: TFont): Integer;
var
  DC: HDC;
  SaveFont: HFont;
  Metrics: TTextMetric;
begin
  DC := GetDC(0);
  SaveFont := SelectObject(DC, Font.Handle);
  GetTextMetrics(DC, Metrics);
  SelectObject(DC, SaveFont);
  ReleaseDC(0, DC);
  Result := Metrics.tmHeight;
end;

procedure CutFirstDirectory(var S: string);
var
  Root: Boolean;
  P: Integer;
begin
  if S = '\' then
    S := ''
  else
  begin
    if S[1] = '\' then
    begin
      Root := True;
      Delete(S, 1, 1);
    end
    else
      Root := False;
    if S[1] = '.' then
      Delete(S, 1, 4);
    P := AnsiPos('\', S);
    if P <> 0 then
    begin
      Delete(S, 1, P);
      S := '...\' + S;
    end
    else
      S := '';
    if Root then
      S := '\' + S;
  end;
end;

function MinimizeName(const Filename: string; Canvas: TCanvas; MaxLen: Integer): string;
var
  Drive: string;
  Dir: string;
  Name: string;
begin
  Result := FileName;
  Dir := ExtractFilePath(Result);
  Name := ExtractFileName(Result);

  if (Length(Dir) >= 2) and (Dir[2] = ':') then
  begin
    Drive := Copy(Dir, 1, 2);
    Delete(Dir, 1, 2);
  end
  else
    Drive := '';
  while ((Dir <> '') or (Drive <> '')) and (Canvas.TextWidth(Result) > MaxLen) do
  begin
    if Dir = '\...\' then
    begin
      Drive := '';
      Dir := '...\';
    end
    else if Dir = '' then
      Drive := ''
    else
      CutFirstDirectory(Dir);
    Result := Drive + Dir + Name;
  end;
end;

function VolumeID(DriveChar: Char): string;
var
  OldErrorMode: Integer;
  Buf: array[0..MAX_PATH - 1] of Char;
  NotUsed, VolFlags: DWORD;
begin
  OldErrorMode := SetErrorMode(SEM_FAILCRITICALERRORS);
  try
    if GetVolumeInformation(PChar(DriveChar + ':\'), Buf, SizeOf(Buf),
      nil, NotUsed, VolFlags, nil, 0) then
      Result := Buf
    else
      Result := '';
    if DriveChar < 'a' then
      Result := AnsiUpperCaseFileName(Result)
    else
      Result := AnsiLowerCaseFileName(Result);
  finally
    SetErrorMode(OldErrorMode);
  end;
end;

function NetworkVolume(DriveChar: Char): string;
var
  Buf: array[0..MAX_PATH - 1] of Char;
  DriveStr: array[0..2] of Char;
  BufferSize: DWORD;
begin
  BufferSize := SizeOf(Buf);
  DriveStr[0] := UpCase(DriveChar);
  DriveStr[1] := ':';
  DriveStr[2] := #0;
  if WNetGetConnection(DriveStr, Buf, BufferSize) = WN_SUCCESS then
  begin
    SetString(Result, Buf, BufferSize);
    if DriveChar < 'a' then
      Result := AnsiUpperCaseFileName(Result)
    else
      Result := AnsiLowerCaseFileName(Result);
  end
  else
    Result := VolumeID(DriveChar);
end;

{ TXDriveComboBox }

constructor TXDriveComboBox.Create(AOwner: TComponent);
var
  Temp: string;
begin
  inherited Create(AOwner);
  Style := csOwnerDrawFixed;
  ReadIcons;
  Temp := GetCurrentDir;
  if (Length(Temp) >= 2) and (Temp[2] = ':') then
    FDrive := Temp[1] { make default drive selected }
  else
    FDrive := #0;
  ResetItemHeight;
end;

destructor TXDriveComboBox.Destroy;
begin
  FreeIcons;
  inherited Destroy;
end;

procedure TXDriveComboBox.BuildList;
var
  DriveNum: Integer;
  DriveChar: Char;
  DriveType: TXDriveType;
  DriveBits: set of 0..25;

  procedure AddDrive(const VolName: string; Obj: TObject);
  begin
    Items.AddObject(Format('%s: %s', [DriveChar, VolName]), Obj);
  end;

begin
  { fill list }
  Clear;
  Longint(DriveBits) := GetLogicalDrives;
  for DriveNum := 0 to 25 do
  begin
    if not (DriveNum in DriveBits) then Continue;
    DriveChar := Char(DriveNum + Ord('a'));
    DriveType := TXDriveType(GetDriveType(PChar(DriveChar + ':\')));
    if TextCase = tcUpperCase then
      DriveChar := Upcase(DriveChar);

    case DriveType of
      dtFloppy: Items.AddObject(DriveChar + ':', Pointer(dcFloppyBMP));
      dtNetwork: AddDrive(NetworkVolume(DriveChar), Pointer(dcNetworkBMP));
      dtCDROM: AddDrive(VolumeID(DriveChar), Pointer(dcCDROMBMP));
      dtRAM: AddDrive(VolumeID(DriveChar), Pointer(dcRAMBMP));
    else
      { Default to fixed for everything else }
      AddDrive(VolumeID(DriveChar), Pointer(dcFixedBMP));
    end;
  end;
end;

procedure TXDriveComboBox.SetDrive(NewDrive: Char);
var
  SaveDrive: Char;
  Item: Integer;
  drv: string;
begin
  if (ItemIndex < 0) or (UpCase(NewDrive) <> UpCase(FDrive)) then
  begin
    SaveDrive := FDrive;
    try
      if NewDrive = #0 then
      begin
        FDrive := NewDrive;
        ItemIndex := -1;
      end
      else
      begin
        FDrive := UpCase(NewDrive);
        if TextCase = tcLowerCase then
          Inc(FDrive, 32);

        { change selected item }
        for Item := 0 to Items.Count - 1 do
        begin
          drv := Items[Item];
          if (UpCase(drv[1]) = UpCase(FDrive)) and (drv[2] = ':') then
          begin
            ItemIndex := Item;
            break;
          end;
        end;
      end;
      if FDirList <> nil then FDirList.DriveChange(Drive);
      Change;
    except
      FDrive := SaveDrive;
      raise;
    end;
  end;
end;

procedure TXDriveComboBox.SetTextCase(XTextCase: TXTextCase);
var
  OldDrive: Char;
begin
  FTextCase := XTextCase;
  OldDrive := FDrive;
  BuildList;
  SetDrive(OldDrive);
end;

procedure TXDriveComboBox.SetDirListBox(Value: TXDirectoryListBox);
begin
  if FDirList <> nil then FDirList.FDriveCombo := nil;
  FDirList := Value;
  if FDirList <> nil then
  begin
    FDirList.FDriveCombo := Self;
    FDirList.FreeNotification(Self);
  end;
end;

procedure TXDriveComboBox.CreateWnd;
begin
  inherited CreateWnd;
  BuildList;
  SetDrive(FDrive);
end;

procedure TXDriveComboBox.DrawItem(Index: Integer; Rect: TRect;
  State: TOwnerDrawState);
var
  R: TRect;
begin
  with Canvas do
  begin
    FillRect(Rect);
    if Icons[TXDriveIcons(Items.Objects[Index])] <> nil then
      DrawIconEx(Canvas.Handle, Rect.Left + 4, Rect.Top, Icons[TXDriveIcons(Items.Objects[Index])].Handle,
        ItemHeight, ItemHeight, 0, Brush.Handle, DI_NORMAL);
    { uses DrawText instead of TextOut in order to get clipping against
      the combo box button }
    R := Rect;
    Inc(R.Left, ItemHeight + 8);
    DrawText(Canvas.Handle, PChar(Items[Index]), -1, R, DT_SINGLELINE or DT_VCENTER or DT_NOPREFIX);
  end;
end;

procedure TXDriveComboBox.Click;
begin
  inherited Click;
  if ItemIndex >= 0 then
  begin
    try
      Drive := Items[ItemIndex][1];
    except
      ItemIndex := -1;
      raise;
    end;
  end;
end;

procedure TXDriveComboBox.CMFontChanged(var Message: TMessage);
begin
  inherited;
  ResetItemHeight;
  RecreateWnd;
end;

procedure TXDriveComboBox.ResetItemHeight;
var
  H: Integer;
begin
  H := GetItemHeight(Font);
  if H < DriveComboMinHeight then H := DriveComboMinHeight;
  ItemHeight := H;
end;

procedure TXDriveComboBox.CMColorChanged(var Message: TMessage);
begin
  ReadIcons;
  inherited;
end;

procedure TXDriveComboBox.CMSysColorChange(var Message: TMessage);
begin
  ReadIcons;
  inherited;
end;

procedure TXDriveComboBox.ReadIcons;
var
  I: TXDriveIcons;
begin
  FreeIcons;
  for I := Low(TXDriveIcons) to High(TXDriveIcons) do
  begin
    Icons[I] := TIcon.Create;
    Icons[I].Handle := LoadImage(HInstance, cDriveIcons[I], IMAGE_ICON, ItemHeight, ItemHeight, LR_DEFAULTCOLOR);
  end;
end;

procedure TXDriveComboBox.FreeIcons;
var
  I: TXDriveIcons;
begin
  for I := Low(I) to High(I) do
    if Assigned(Icons[I]) then
    begin
      Icons[I].Free;
      Icons[I] := nil;
    end;
end;

procedure TXDriveComboBox.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = FDirList) then
    FDirList := nil;
end;

{ TNewDirectoryListBox }

function DirLevel(PathName: string): Integer;
var
  P: PChar;
begin
  Result := 0;
  if (PathName <> '') and
    (AnsiLastChar(PathName)^ <> '\') then
    Pathname := PathName + '\';
  if (Length(PathName) >= 2) and (PathName[1] = '\') and (PathName[2] = '\') then
    Result := -3; { handle UNC names appropriately }
  P := AnsiStrScan(PChar(PathName), '\');
  while P <> nil do
  begin
    Inc(Result);
    Inc(P);
    P := AnsiStrScan(P, '\');
  end;
end;

constructor TXDirectoryListBox.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Width := 145;
  Style := lbOwnerDrawFixed;
  Sorted := False;
  ReadIcons;
  FDirectory := GetCurrentDir; { initially use current dir on default drive }
  ResetItemHeight;
end;

destructor TXDirectoryListBox.Destroy;
begin
  FreeIcons;
  inherited Destroy;
end;

procedure TXDirectoryListBox.DriveChange(NewDrive: Char);
begin
  if UpCase(NewDrive) <> UpCase(Drive) then
  begin
    if NewDrive <> #0 then
    begin
      ChDir(NewDrive + ':');
      FDirectory := GetCurrentDir; { store correct directory name }
    end;
    if not FInSetDir then
    begin
      BuildList;
      Change;
    end;
  end;
end;

procedure TXDirectoryListBox.SetDirLabel(Value: TXPathLabel);
begin
  FDirLabel := Value;
  if Value <> nil then Value.FreeNotification(Self);
  SetDirLabelCaption;
end;

procedure TXDirectoryListBox.SetDir(const NewDirectory: string);
begin
     { go to old directory first, in case of incomplete pathname
       and curdir changed - probably not necessary }
  SetCurrentDir(FDirectory);

  ChDir(NewDirectory); { exception raised if invalid dir }
  FDirectory := GetCurrentDir; { store correct directory name }
  BuildList;
  Change;
end;

procedure TXDirectoryListBox.OpenCurrent;
begin
  Directory := GetItemPath(ItemIndex);
end;

procedure TXDirectoryListBox.Refresh;
begin
  BuildList;
  Change;
end;

function TXDirectoryListBox.DisplayCase(const S: string): string;
begin
  if (S = '') or
    (FPreserveCase and ((S[Length(S)] = '\') or (S <> AnsiUpperCase(S)))) or
  FCaseSensitive then
    Result := S
  else
    Result := AnsiLowerCase(S);
end;

function TXDirectoryListBox.FileCompareText(const A, B: string): Integer;
begin
  if FCaseSensitive then
    Result := AnsiCompareStr(A, B)
  else
    Result := AnsiCompareFileName(A, B);
end;

{
    Reads all directories in ParentDirectory, adds their paths to
    DirectoryList,and returns the number added
  }

function TXDirectoryListBox.ReadDirectoryNames(const ParentDirectory: string;
  DirectoryList: TStringList): Integer;
var
  Status: Integer;
  SearchRec: TSearchRec;
begin
  Result := 0;
  Status := FindFirst(Slash(ParentDirectory, '*.*'), faDirectory, SearchRec);
  try
    while Status = 0 do
    begin
      if (SearchRec.Attr and faDirectory = faDirectory) then
      begin
        if (SearchRec.Name <> '.') and (SearchRec.Name <> '..') then
        begin
          DirectoryList.Add(SearchRec.Name);
          Inc(Result);
        end;
      end;
      Status := FindNext(SearchRec);
    end;
  finally
    FindClose(SearchRec);
  end;
end;

procedure TXDirectoryListBox.BuildList;
var
  TempPath: string;
  DirName: string;
  IndentLevel, BackSlashPos, i: Integer;
  VolFlags, NotUsed: DWORD;
  Siblings: TStringList;
  NewSelect: Integer;
  Root: string;
begin
  Items.BeginUpdate;
  try
    Items.Clear;
    Root := ExtractFileDrive(Directory) + '\';
    if not GetVolumeInformation(PChar(Root), nil, 0, nil, NotUsed, VolFlags, nil, 0) then
      Exit;
    FPreserveCase := VolFlags and (FS_CASE_IS_PRESERVED or FS_CASE_SENSITIVE) <> 0;
    FCaseSensitive := (VolFlags and FS_CASE_SENSITIVE) <> 0;
    IndentLevel := 0;
    if (Length(Root) >= 2) and (Root[2] = '\') then { UNC name }
    begin
      TempPath := Copy(Directory, Length(Root) + 1, Maxint);
      if TempPath = '' then
        Items.AddObject(Root, Pointer(dcCurrentBMP))
      else
        Items.AddObject(Root, Pointer(dcOpenedBMP));
      Inc(IndentLevel);
    end
    else
      TempPath := Directory;
    if TempPath <> '' then
    begin
      if AnsiLastChar(TempPath)^ <> '\' then
      begin
        BackSlashPos := AnsiPos('\', TempPath);
        while BackSlashPos <> 0 do
        begin
          DirName := Copy(TempPath, 1, BackSlashPos - 1);
          if IndentLevel = 0 then DirName := DirName + '\';
          Delete(TempPath, 1, BackSlashPos);
          Items.AddObject(DirName, Pointer(dcOpenedBMP));
          Inc(IndentLevel);
          BackSlashPos := AnsiPos('\', TempPath);
        end;
      end;
      Items.AddObject(TempPath, Pointer(dcCurrentBMP));
    end;
    NewSelect := Items.Count - 1;
    Siblings := TStringList.Create;
    try
      Siblings.Sorted := True;
        { read all the dir names into Siblings }
      ReadDirectoryNames(Directory, Siblings);
      for i := 0 to Siblings.Count - 1 do
        Items.AddObject(Siblings[i], Pointer(dcClosedBMP))
    finally
      Siblings.Free;
    end;
  finally
    Items.EndUpdate;
  end;
  if HandleAllocated then
    ItemIndex := NewSelect;
end;

procedure TXDirectoryListBox.ReadIcons;
var
  I: TXDirectoryIcons;
begin
  FreeIcons;
  for I := Low(TXDirectoryIcons) to High(TXDirectoryIcons) do
  begin
    Icons[I] := TIcon.Create;
    Icons[I].Handle := LoadImage(HInstance, cDirectoryIcons[I], IMAGE_ICON, ItemHeight, ItemHeight, LR_DEFAULTCOLOR);
  end;
end;

procedure TXDirectoryListBox.FreeIcons;
var
  I: TXDirectoryIcons;
begin
  for I := Low(I) to High(I) do
    if Assigned(Icons[I]) then
    begin
      Icons[I].Free;
      Icons[I] := nil;
    end;
end;

procedure TXDirectoryListBox.DblClick;
begin
  inherited DblClick;
  OpenCurrent;
end;

procedure TXDirectoryListBox.Change;
begin
  if FFileList <> nil then FFileList.SetDirectory(Directory);
  SetDirLabelCaption;
  if Assigned(FOnChange) then FOnChange(Self);
end;

procedure TXDirectoryListBox.DrawItem(Index: Integer; Rect: TRect;
  State: TOwnerDrawState);
var
  Icon: TIcon;
  dirOffset: Integer;
  Ico: TXDirectoryIcons;
  R: TRect;
begin
  with Canvas do
  begin
    FillRect(Rect);
    dirOffset := Index * 4 + 4; {add 4 for spacing}
    Ico := TXDirectoryIcons(Items.Objects[Index]);
    Icon := Icons[Ico];
    if Icon <> nil then
    begin
      if Ico = dcClosedBMP then
        dirOffset := DirLevel(Directory) * 4 + 4;
      DrawIconEx(Canvas.Handle, Rect.Left + dirOffset, Rect.Top, Icon.Handle,
        ItemHeight, ItemHeight, 0, Brush.Handle, DI_NORMAL);
    end;
    R := Classes.Rect(Rect.Left + ItemHeight + dirOffset + 4, Rect.Top, Rect.Right, Rect.Bottom);
    DrawText(Handle, PChar(DisplayCase(Items[Index])), -1, R, DT_SINGLELINE or DT_VCENTER or DT_NOPREFIX);
  end;
end;

function TXDirectoryListBox.GetItemPath(Index: Integer): string;
var
  CurDir: string;
  i, j: Integer;
  Bitmap: TXDirectoryIcons;
begin
  Result := '';
  if Index < Items.Count then
  begin
    CurDir := Directory;
    Bitmap := TXDirectoryIcons(Items.Objects[Index]);
    if Index = 0 then
      Result := ExtractFileDrive(CurDir) + '\'
    else if Bitmap = dcClosedBMP then
      Result := Slash(CurDir, Items[Index])
    else if Bitmap = dcCurrentBMP then
      Result := CurDir
    else
    begin
      i := 0;
      j := 0;
      Delete(CurDir, 1, Length(ExtractFileDrive(CurDir)));
      while j <> (Index + 1) do
      begin
        Inc(i);
        if i > Length(CurDir) then
          break;
        if CurDir[i] in LeadBytes then
          Inc(i)
        else if CurDir[i] = '\' then
          Inc(j);
      end;
      Result := ExtractFileDrive(Directory) + Copy(CurDir, 1, i - 1);
    end;
  end;
end;

procedure TXDirectoryListBox.CreateWnd;
begin
  inherited CreateWnd;
  BuildList;
end;

procedure TXDirectoryListBox.CMFontChanged(var Message: TMessage);
begin
  inherited;
  ResetItemHeight;
end;

procedure TXDirectoryListBox.ResetItemHeight;
var
  H: Integer;
begin
  H := GetItemHeight(Font);
  if H < DirListMinHeight then H := DirListMinHeight;
  ItemHeight := H;
end;

procedure TXDirectoryListBox.CMColorChanged(var Message: TMessage);
begin
  ReadIcons;
  inherited;
end;

procedure TXDirectoryListBox.CMSysColorChange(var Message: TMessage);
begin
  ReadIcons;
  inherited;
end;

function TXDirectoryListBox.GetDrive: Char;
begin
  Result := FDirectory[1];
end;

procedure TXDirectoryListBox.SetFileListBox(Value: TXFileListBox);
begin
  if FFileList <> nil then FFileList.FDirList := nil;
  FFileList := Value;
  if FFileList <> nil then
  begin
    FFileList.FDirList := Self;
    FFileList.FreeNotification(Self);
  end;
end;

procedure TXDirectoryListBox.SetDrive(Value: Char);
begin
  if UpCase(Value) <> UpCase(Drive) then
    SetDirectory(Format('%s:', [Value]));
end;

procedure TXDirectoryListBox.SetDirectory(const NewDirectory: string);
var
  DirPart: string;
  FilePart: string;
  NewDrive: Char;
begin
  if NewDirectory = '' then Exit;
  if (FileCompareText(NewDirectory, Directory) = 0) then Exit;
  ProcessPath(NewDirectory, NewDrive, DirPart, FilePart);
  try
    if Drive <> NewDrive then
    begin
      FInSetDir := True;
      if (FDriveCombo <> nil) then
        FDriveCombo.Drive := NewDrive
      else
        DriveChange(NewDrive);
    end;
  finally
    FInSetDir := False;
  end;
  SetDir(DirPart);
end;

procedure TXDirectoryListBox.KeyPress(var Key: Char);
begin
  inherited KeyPress(Key);
  if Ord(Key) = VK_RETURN then
    OpenCurrent;
end;

procedure TXDirectoryListBox.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) then
  begin
    if (AComponent = FFileList) then
      FFileList := nil
    else if (AComponent = FDriveCombo) then
      FDriveCombo := nil
    else if (AComponent = FDirLabel) then
      FDirLabel := nil;
  end;
end;

procedure TXDirectoryListBox.SetDirLabelCaption;
begin
  if FDirLabel <> nil then
    FDirLabel.Caption := Directory;
end;

{ ************ TXFileListBox ************ }

constructor TXFileListBox.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
//  Width := 145;
  GetDir(0, FDirectory); { initially use current dir on default drive }
  FMask := DefaultMask; { default file mask is all }
  MultiSelect := False; { default is not multi-select }
  FLastSel := -1;
  Style := lbOwnerDrawFixed;
  ResetItemHeight;
// my own
  ItemHeight := GetSystemMetrics(SM_CXSMICON);
  Height := 196;
  Sorted := false;
  FIcons := TStringList.Create;
  FSortList := TSortStringList.Create;
  FSortList.CompareFunction := CompareFileName;
  FCompareFunctions[fiFileName] := CompareFileName;
  FCompareFunctions[fiSize] := CompareSize;
  FCompareFunctions[fiDate] := CompareDate;
  FCompareFunctions[fiTime] := CompareTime;
  FCompareFunctions[fiAttribute] := CompareAttribute;
  FSortBy := fiFileName;
  FSortOrder := soAscending;
  FSortOrderMultiplier := 1;
  FWidthFileName := cWidthFileName;
  FWidthSize := cWidthSize;
  FWidthDate := cWidthDate;
  FWidthTime := cWidthTime;
  FWidthAttribute := cWidthAttribute;
  FOpenParentFolders := true;
  FOpenFolders := true;
  FOpenFiles := true;
  FShowItems := cDefaultShowItems;
  FShowParentFolder := true;
  FShowFolderAnimation := true;
  FBeepErrors := true;
  FDateFormat := cDefaultDateFormat;
  FTimeFormat := cDefaultTimeFormat;
  FDirName := cDefaultDirName;
  FFileType := cDefaultFileType;
  FCached := true;
  FShellIcons := true;
  FShowGlyphs := true;
  FAutoWidth := true;
  AdjustWidth;
  LoadStdIcons;
end;

destructor TXFileListBox.Destroy;
begin
//  FreeData; // causes EInvalidOperation error - I don't know why...
  FreeIcons;
  FreeStdIcons;
  FSortList.Free;
  inherited Destroy;
end;

function TXFileListBox.FormatDirName(Dir: string): string;
begin
  Result := Format('[%s]', [Dir]);
end;

function TXFileListBox.UnFormatDirName(Dir: string): string;
begin
  Result := Copy(Dir, 2, Length(Dir) - 2);
end;

function TXFileListBox.GetFilePath: string;
begin
  Result := '';
  if GetFileName <> '' then
    if AnsiLastChar(FDirectory)^ <> '\' then
      Result := FDirectory + '\' + GetFileName
    else
      Result := FDirectory + GetFileName;
end;
{function TXFileListBox.GetFilePath: string;
begin
  Result := '';
  if GetFileName <> '' then
    Result := SlashSep(FDirectory, GetFileName);
end;}

procedure TXFileListBox.LoadStdIcons;
var
  Index: integer;
begin
  for Index := Abs(iiDefault) to Abs(iiShortcut) do
  begin
    FStdIcons[Index] := TIcon.Create;
    FStdIcons[Index].Handle := LoadImage(HInstance, cStdIcons[Index], IMAGE_ICON, ItemHeight, ItemHeight, LR_DEFAULTCOLOR);
  end;
end;

procedure TXFileListBox.FreeData;
var
  Index: integer;
begin
  for Index := 1 to Items.Count do
  begin
    Dispose(PData(Items.Objects[Index - 1]));
    Items.Objects[Index - 1] := nil;
  end;
end;

procedure TXFileListBox.FreeStdIcons;
var
  Index: integer;
begin
  for Index := Abs(iiDefault) to Abs(iiShortcut) do
    FStdIcons[Index].Free;
end;

procedure TXFileListBox.FreeIcons;
var
  Index: integer;
begin
  for Index := 1 to FIcons.Count do
    FIcons.Objects[Index - 1].Free;
  FIcons.Free;
end;

procedure TXFileListBox.ReadFileNames;
var
  MaskPtr: PChar;
  Ptr: PChar;
  AttrWord: integer;
  FileInfo: TSearchRec;
  SaveCursor: TCursor;
  Extension: string;
  IconType: integer;
  DataItem: PData;
  procedure SetData(Value: integer);
  begin
    New(DataItem);
    with DataItem^ do
    begin
      IconIndex := Value;
      Date := FileDateToDateTime(FileInfo.Time);
      Size := FileInfo.Size;
      Attribute := FileInfo.Attr;
      FileNameLength := Length(FileInfo.Name); // for quick sorting
    end;
  end;
begin
  { if no handle allocated yet, this call will force
    one to be allocated incorrectly (i.e. at the wrong time.
    In due time, one will be allocated appropriately.  }
  if HandleAllocated then
  begin
    AttrWord := FileTypeToAttr(FileType) or DDL_READWRITE; //Set attribute flags based on FileType
    ChDir(FDirectory); { go to the directory we want }
    FParentFolder := -1;
    FFirstFolder := -1;
    FLastFolder := -1;
    FFoldersCount := 0;
    FFilesCount := 0;
    FSortList.BeginUpdate;
    SaveCursor := Screen.Cursor;
    Screen.Cursor := crHourglass;
    try
      { read all DIRECTORIES }
      if (xftDirectory in FileType) and (FindFirst('*.*', faAnyFile - faVolumeID, FileInfo) = 0) then
      begin
        repeat
          if ((FileInfo.Attr and faDirectory) <> 0) and (FileInfo.Name <> '.') then
            if FileInfo.Name = '..' then
            begin
              if ShowParentFolder then
              begin
{$IFNDEF NO_CUSTOM_FILTER}
                if not(Assigned(FOnAcceptFile) and not FOnAcceptFile(itParentFolder, FileInfo)) then
{$ENDIF}
                begin
                  FileInfo.Name := FormatDirName(FileInfo.Name);
                  SetData(iiFolderParent);
                  FParentFolder := FSortList.AddObject(FileInfo.Name, Pointer(DataItem));
                  Inc(FFoldersCount);
                end;
              end;
            end
            else
            begin
{$IFNDEF NO_CUSTOM_FILTER}
              if not(Assigned(FOnAcceptFile) and not FOnAcceptFile(itFolder, FileInfo)) then
{$ENDIF}
              begin
                if FileInfo.Attr and faHidden <> 0 then
                  if FileInfo.Attr and faSysFile <> 0 then
                    IconType := iiFolderHiddenSystem
                  else
                    IconType := iiFolderHidden
                else if FileInfo.Attr and faSysFile <> 0 then
                  IconType := iiFolderSystem
                else
                  IconType := iiFolder;
                FileInfo.Name := FormatDirName(FileInfo.Name);
                SetData(IconType);
                FLastFolder := FSortList.AddObject(FileInfo.Name, Pointer(DataItem));
                if FFirstFolder = -1 then
                  FFirstFolder := FLastFolder;
                Inc(FFoldersCount);
              end;
            end;
          if FLastFolder = 100 then
            Screen.Cursor := crHourglass;
        until FindNext(FileInfo) <> 0;
        FindClose(FileInfo);
      end;
      { read FILES }
      MaskPtr := PChar(FMask);
      while MaskPtr <> nil do
      begin
        Ptr := StrScan(MaskPtr, ';');
        if Ptr <> nil then
          Ptr^ := #0;
        if FindFirst(MaskPtr, AttrWord, FileInfo) = 0 then
        begin
          repeat { exclude normal files if ftNormal not set }
            if (xftNormal in FileType) or (FileInfo.Attr and AttrWord <> 0) then
            begin
              if FileInfo.Attr and faDirectory <> 0 then
              begin
              end
              else
              begin
{$IFNDEF NO_CUSTOM_FILTER}
                if not(Assigned(FOnAcceptFile) and not FOnAcceptFile(itFile, FileInfo)) then
{$ENDIF}
                begin
                  if ShellIcons then
                    IconType := iiUnknown
                  else
                  begin
                    Extension := AnsiLowerCase(ExtractFileExt(FileInfo.Name));
                    if (Extension = '.exe') or (Extension = '.com') or (Extension = '.bat') then
                      IconType := iiExeWin
                    else if Extension = '.pif' then
                      IconType := iiExeDos
                    else if Extension = '.lnk' then
                      IconType := iiShortcut
                    else if (Extension = '.zip') or (Extension = '.rar') or (Extension = '.arj') or (Extension = '.jar') or (Extension = '.cab') then
                      IconType := iiCompress
                    else if ((FileInfo.Attr and faHidden) <> 0) or ((FileInfo.Attr and faSysFile) <> 0) then
                      IconType := iiDefaultHidden
                    else
                      IconType := iiDefault;
                  end;
                  SetData(IconType);
                  FSortList.AddObject(FileInfo.Name, pointer(DataItem));
                  Inc(FFilesCount);
                end;
              end;
            end;
          until FindNext(FileInfo) <> 0;
          FindClose(FileInfo);
        end;
        if Ptr <> nil then
        begin
          Ptr^ := ';';
          Inc(Ptr);
        end;
        MaskPtr := Ptr;
      end;
      Sort;
      Items.BeginUpdate;
      FreeData; //clear data elements
      Items.Assign(FSortList);
      Items.EndUpdate;
      FSortList.Clear;
      FSortList.EndUpdate;
    finally
      Screen.Cursor := SaveCursor;
    end;
    Change;
    if Assigned(FOnUpdateContent) then
      FOnUpdateContent;
  end;
end;

function TXFileListBox.GetData(StringList: TStrings; Index: integer): PData;
begin
  if (Index >= 0) and (Index < StringList.Count) then
    Result := PData(StringList.Objects[Index])
  else
    Result := nil; // possible problem
end;

function TXFileListBox.GetData(Index: integer): PData;
begin
  if (Index >= 0) and (Index < Items.Count) then
    Result := PData(Items.Objects[Index])
  else
    Result := nil; // possible problem
end;

function TXFileListBox.GetType(IconIndex: integer): TItemType; // index is iiUnknown, iiFolder..
begin
  case IconIndex of
    iiFolderParent: Result := itParentFolder;
    iiFolder, iiFolderOpen, iiFolderShared, iiFolderHidden,
      iiFolderSystem, iiFolderHiddenSystem: Result := itFolder;
  else
    Result := itFile;
  end;
end;

function TXFileListBox.GetItemType(StringList: TStrings; Index: integer): TItemType;
begin
  if (StringList.Count > 0) and ((Index >= 0) and (Index < StringList.Count)) then
    Result := GetType(GetData(StringList, Index)^.IconIndex)
  else
    Result := itUnknown;
end;

function TXFileListBox.GetItemType(Index: integer): TItemType;
begin
  Result := GetItemType(Items, Index);
end;

function TXFileListBox.GetItemType: TItemType;
begin
  if ItemIndex <> -1 then
    Result := GetItemType(ItemIndex)
  else
    Result := itUnknown;
end;

function TXFileListBox.GenerateIcon(Index: integer): TIcon;
var
  Extension: string;
  FullName: string;
  Icon: TIcon;
  IconIndex: integer;
  Position: integer;
  ShellFileInfo: TSHFileInfo;
begin
  IconIndex := GetData(Index)^.IconIndex;
  case IconIndex of
    iiUnknown:
      begin
        Extension := AnsiLowerCase(ExtractFileExt(Items[Index]));
        if (Extension = '.exe') or (Extension = '.com') or (Extension = '.pif')
          or (Extension = '.ico') or (Extension = '.icl') or (Extension = '.cpl')
          or (Extension = '.cur') or (Extension = '.ani') or (Extension = '.lnk') then
          FullName := Slash(Directory, Items[Index])
        else
          FullName := Extension;
        Position := FIcons.IndexOf(FullName);
        if Position <> -1 then
          Result := TIcon(FIcons.Objects[Position])
        else
        begin
          Icon := TIcon.Create;
          ShGetFileInfo(PChar(Slash(Directory, Items[Index])), 0, ShellFileInfo, SizeOf(ShellFileInfo),
            SHGFI_ICON or SHGFI_SMALLICON);
          Icon.Handle := ShellFileInfo.hIcon;
          PData(Items.Objects[Index])^.IconIndex := FIcons.AddObject(FullName, Icon);
          Result := Icon;
        end;
      end;
    iiDefault: Result := FStdIcons[Abs(iiDefault)];
    iiDefaultHidden: Result := FStdIcons[Abs(iiDefaultHidden)];
    iiFolder: Result := FStdIcons[Abs(iiFolder)];
    iiFolderOpen: Result := FStdIcons[Abs(iiFolderOpen)];
    iiFolderShared: Result := FStdIcons[Abs(iiFolderShared)];
    iiFolderHidden: Result := FStdIcons[Abs(iiFolderHidden)];
    iiFolderSystem: Result := FStdIcons[Abs(iiFolderSystem)];
    iiFolderHiddenSystem: Result := FStdIcons[Abs(iiFolderHiddenSystem)];
    iiFolderParent: Result := FStdIcons[Abs(iiFolderParent)];
    iiExeDos: Result := FStdIcons[Abs(iiExeDos)];
    iiExeWin: Result := FStdIcons[Abs(iiExeWin)];
    iiCompress: Result := FStdIcons[Abs(iiCompress)];
    iiCompressOpen: Result := FStdIcons[Abs(iiCompressOpen)];
    iiShortcut: Result := FStdIcons[Abs(iiShortcut)];
  else
    Result := TIcon(FIcons.Objects[IconIndex]);
  end;
end;

procedure TXFileListBox.DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState);
var
  Icon: TIcon;
  Offset: Integer;
  function MinimizeFileName(FileName: string; MaxLen: Integer): string;
  begin
    Result := FileName;
    while (Canvas.TextWidth(Result) > MaxLen) do
    begin
      Delete(Result, Length(Result) - 3, 4);
      Result := Result + '...'; //
    end;
  end;
  function FormatDateTime(const FormatStr: string; Date: TDateTime): string;
  begin
    DateTimeToString(Result, FormatStr, Date);
  end;
  function FormatSize(Value: integer): string;
  begin
    FmtStr(Result, '%d', [Value]);
    if Length(Result) > 6 then
    begin
      Insert(' ', Result, Length(Result) - 5);
      Insert(' ', Result, Length(Result) - 2);
    end
    else if Length(Result) > 3 then
    begin
      Insert(' ', Result, Length(Result) - 2);
    end;
  end;
begin
  with Canvas do
  begin
    FillRect(Rect);
    Offset := cOffsetWidth;
    if ShowGlyphs then
    begin
      Icon := GenerateIcon(Index);
      if Assigned(Icon) then
      begin
        DrawIconEx(Self.Canvas.Handle, Rect.Left + cOffsetWidth, Rect.Top, Icon.Handle, ItemHeight, ItemHeight, 0,
          Self.Canvas.Brush.Handle, DI_NORMAL);
        Inc(Offset, ItemHeight + cOffsetWidth);
      end;
    end;
    if fiFileName in ShowItems then
    begin
      Rect.Left := Rect.Left + Offset;
      Rect.Right := Rect.Left + WidthFileName;
      SetTextAlign(Canvas.Handle, TA_LEFT);
      TextRect(Rect, Rect.Left, Rect.Top, MinimizeFileName(Items[Index], Rect.Right - Rect.Left));
    end;
    if fiSize in ShowItems then
    begin
      Rect.Left := Rect.Right + cSpaceWidth;
      Rect.Right := Rect.Left + WidthSize;
      SetTextAlign(Canvas.Handle, TA_RIGHT);
      if GetItemType(Index) = itFile then
        TextRect(Rect, Rect.Right, Rect.Top, FormatSize(GetData(Index)^.Size))
      else
        TextRect(Rect, Rect.Right, Rect.Top, DirName);
    end;
    if fiDate in ShowItems then
    begin
      Rect.Left := Rect.Right + cSpaceWidth;
      Rect.Right := Rect.Left + WidthDate;
      SetTextAlign(Canvas.Handle, TA_LEFT);
      TextRect(Rect, Rect.Left, Rect.Top, FormatDateTime(DateFormat, GetData(Index)^.Date));
    end;
    if fiTime in ShowItems then
    begin
      Rect.Left := Rect.Right + cSpaceWidth;
      Rect.Right := Rect.Left + WidthTime;
      SetTextAlign(Canvas.Handle, TA_LEFT);
      TextRect(Rect, Rect.Left, Rect.Top, FormatDateTime(TimeFormat, GetData(Index)^.Date));
    end;
    if fiAttribute in ShowItems then
    begin
      Rect.Left := Rect.Right + cSpaceWidth;
      Rect.Right := Rect.Left + WidthAttribute;
      SetTextAlign(Canvas.Handle, TA_LEFT);
      TextRect(Rect, Rect.Left, Rect.Top, AttrToStr(GetData(Index)^.Attribute));
    end;
  end;
end;

procedure TXFileListBox.AdjustWidth;
var
  Index: integer;
begin
  if AutoWidth then
  begin
    Index := (2 * cOffsetWidth) + GetSystemMetrics(SM_CXVSCROLL) + (2 * GetSystemMetrics(SM_CXEDGE));
    if ShowGlyphs then
      Inc(Index, cOffsetWidth + ItemHeight);
    if fiFileName in ShowItems then
      Inc(Index, WidthFileName);
    if fiSize in ShowItems then
      Inc(Index, WidthSize + cSpaceWidth);
    if fiDate in ShowItems then
      Inc(Index, WidthDate + cSpaceWidth);
    if fiTime in ShowItems then
      Inc(Index, WidthTime + cSpaceWidth);
    if fiAttribute in ShowItems then
      Inc(Index, WidthAttribute + cSpaceWidth);
    if Index <> Width then
    begin
      Width := Index;
      Invalidate;
      Repaint;
    end;
  end;
end;

procedure TXFileListBox.OpenParentFolder;
var
  Previous: string;
  Index: integer;
begin
  if (Length(Directory) > 3) and (OpenParentFolders) then
  begin
    Items.BeginUpdate;
    Previous := FormatDirName(ExtractFileName(Directory));
    Directory := ExtractFileDir(Directory);
    if xftDirectory in FileType then
    begin
      Index := Items.IndexOf(Previous);
      if Index <> -1 then
      begin
        ItemIndex := Index;
        if MultiSelect then Selected[Index] := true;
      end
      else
      begin
        ItemIndex := 0;
        if MultiSelect then Selected[0] := true;
      end;
    end;
    Items.EndUpdate;
  end;
end;

{$WARNINGS OFF}

procedure TXFileListBox.OpenFolder;
var
  Prev: string;
  PrevIndex: integer;
begin
  if OpenFolders then
  begin
    if ShowFolderAnimation then
    begin
      PrevIndex := PData(Items.Objects[ItemIndex])^.IconIndex;
      PData(Items.Objects[ItemIndex])^.IconIndex := iiFolderOpen;
      DrawItem(ItemIndex, ItemRect(ItemIndex), [odFocused]);
    end;
    Items.BeginUpdate;
    Prev := Directory;
    ApplyFilePath(FileName + '\'); // tricky ?-) way to enter directory
    if (Prev = Directory) then
    begin
      if ShowFolderAnimation then
      begin
        PData(Items.Objects[ItemIndex])^.IconIndex := PrevIndex;
        DrawItem(ItemIndex, ItemRect(ItemIndex), [odFocused]);
        if FBeepErrors then
          MessageBeep(MB_ICONHAND);
      end;
    end
    else
    begin
      ItemIndex := 0;
      if MultiSelect then Selected[0] := true;
    end;
    Items.EndUpdate;
  end;
end;
{$WARNINGS ON}

procedure TXFileListBox.OpenFile;
begin
  if OpenFiles then
    if ShellExecute(MainInstance, nil, PChar(Slash(Directory, Items[ItemIndex])),
      PChar(Parametres), PChar(Directory), SW_SHOWDEFAULT) <= 32 then
      if FBeepErrors then
        MessageBeep(MB_ICONHAND);
  if Assigned(FOnOpenFile) then
    FOnOpenFile(Slash(Directory, Items[ItemIndex]));
end;

procedure TXFileListBox.Open;
begin
  if (Items.Count > 0) and (ItemIndex <> -1) then
    case GetData(ItemIndex)^.IconIndex of
      iiFolderParent:
        OpenParentFolder;
      iiFolder, iiFolderOpen, iiFolderShared, iiFolderHidden, iiFolderSystem, iiFolderHiddenSystem:
        OpenFolder;
    else
      OpenFile;
    end;
end;

procedure TXFileListBox.DblClick;
begin
  Open;
end;

procedure TXFileListBox.KeyDown(var Key: Word; Shift: TShiftState);
begin
  case Key of
    vk_Return:
      begin
        Key := 0;
        Open;
      end;
    vk_Prior:
      if ssCtrl in Shift then
      begin
        Key := 0;
        OpenParentFolder;
      end;
    vk_Next:
      if ssCtrl in Shift then
      begin
        Key := 0;
        Open;
      end;
{    vk_Apps:                                                                    //local menu
      begin
        Beep;
      end;}
  end;
  inherited KeyDown(Key, Shift);
end;

function TXFileListBox.GetFileName(Index: integer): string;
begin
  if (Index >= 0) and (Index < Items.Count) then
  begin
    Result := Items[Index];
    if (GetData(Index)^.IconIndex >= iiFolderHiddenSystem) and (GetData(Index)^.IconIndex <= iiFolder) then
      Result := UnFormatDirName(Result);
  end
  else
    Result := '';
end;

function TXFileListBox.GetFileName: string;
begin
  if (ItemIndex < 0) or (Items.Count = 0) or (Selected[ItemIndex] = false) then
    Result := ''
  else
  begin
    Result := Items[ItemIndex];
    if (GetData(ItemIndex)^.IconIndex >= iiFolderHiddenSystem) and (GetData(ItemIndex)^.IconIndex <= iiFolder) then
      Result := UnFormatDirName(Result);
  end;
end;

procedure TXFileListBox.FWriteAutoWidth(Value: boolean);
begin
  if Value <> FAutoWidth then
  begin
    FAutoWidth := Value;
    if Value then
      AdjustWidth;
  end;
end;

procedure TXFileListBox.FWriteDirLabel(Value: TXPathLabel);
begin
  FDirLabel := Value;
  if Value <> nil then Value.FreeNotification(Self);
  SetDirLabelCaption;
end;

procedure TXFileListBox.SetDirLabelCaption;
begin
  if FDirLabel <> nil then
  begin
    FDirLabel.Caption := Directory;
    if FDirLabelMask then
      FDirLabel.Caption := Slash(FDirLabel.Caption, Mask);
  end;
end;

procedure TXFileListBox.FWriteDirLabelMask(Value: boolean);
begin
  FDirLabelMask := Value;
  SetDirLabelCaption;
end;

procedure TXFileListBox.FWriteFileType(Value: TXFileType);
begin
  if Value <> FFileType then
  begin
    FFileType := Value;
    ReadFileNames;
  end;
end;

procedure TXFileListBox.FWriteShowItems(Value: TShowItems);
begin
  if Value <> FShowItems then
  begin
    FShowItems := Value + [fiFileName];
    AdjustWidth;
  end;
end;

procedure TXFileListBox.FWriteShowParentFolder(Value: boolean);
begin
  if Value <> FShowParentFolder then
  begin
    FShowParentFolder := Value;
    Update;
  end;
end;

procedure TXFileListBox.FWriteCached(Value: boolean);
begin
  FCached := Value;
end;

procedure TXFileListBox.FWriteShellIcons(Value: boolean);
begin
  if Value <> FShellIcons then
  begin
    FShellIcons := Value;
    Update;
  end;
end;

procedure TXFileListBox.FWriteWidthFileName(Value: word);
begin
  FWidthFileName := Value;
  if fiFileName in ShowItems then
    AdjustWidth;
end;

procedure TXFileListBox.FWriteWidthSize(Value: word);
begin
  FWidthSize := Value;
  if fiSize in ShowItems then
    AdjustWidth;
end;

procedure TXFileListBox.FWriteWidthDate(Value: word);
begin
  FWidthDate := Value;
  if fiDate in ShowItems then
    AdjustWidth;
end;

procedure TXFileListBox.FWriteWidthTime(Value: word);
begin
  FWidthTime := Value;
  if fiTime in ShowItems then
    AdjustWidth;
end;

procedure TXFileListBox.FWriteWidthAttribute(Value: word);
begin
  FWidthAttribute := Value;
  if fiAttribute in ShowItems then
    AdjustWidth;
end;

procedure TXFileListBox.FWriteDateFormat(Value: string);
begin
  if FDateFormat <> Value then
  begin
    FDateFormat := Value;
    Invalidate;
    Repaint;
  end;
end;

procedure TXFileListBox.FWriteTimeFormat(Value: string);
begin
  if FTimeFormat <> Value then
  begin
    FTimeFormat := Value;
    Invalidate;
    Repaint;
  end;
end;

procedure TXFileListBox.FWriteDirName(Value: string);
begin
  if FDirName <> Value then
  begin
    FDirName := Value;
    if xftDirectory in FileType then
    begin
      Invalidate;
      Repaint;
    end;
  end;
end;

procedure TXFileListBox.FWriteSortBy(Value: TFileItem);
begin
  if FSortBy <> Value then
  begin
    FSortBy := Value;
    ReSort;
  end;
end;

procedure TXFileListBox.FWriteSortOrder(Value: TSortOrder);
begin
  if FSortOrder <> Value then
  begin
    FSortOrder := Value;
    FSortOrderMultiplier := -1 * FSortOrderMultiplier;
    ReSort;
  end;
end;

procedure TXFileListBox.FWriteSortFolders(Value: TSortFolders);
begin
  if FSortFolders <> Value then
  begin
    FSortFolders := Value;
    ReSort;
  end;
end;

function TXFileListBox.FReadFullFileNames(Index: integer): string;
begin
  Result := Slash(Directory, FileNames[Index]);
end;

function TXFileListBox.FReadFileNames(Index: integer): string;
begin
  if (Index < Items.Count) then
    Result := GetFileName(Index)
  else
    Result := '';
end;

procedure TXFileListBox.FWriteFileNames(Index: integer; Value: string);
begin
  if (Index > 0) and (Index < Items.Count) then
    if ((GetData(Index)^.IconIndex >= iiFolderHiddenSystem) and (GetData(Index)^.IconIndex <= iiFolder)) then
      Items[Index] := FormatDirName(Value)
    else
      Items[Index] := Value;
end;

function TXFileListBox.FReadDate(Index: integer): TDateTime;
begin
  if (Index < Items.Count) then
    Result := GetData(Index)^.Date
  else
    Result := 0;
end;

function TXFileListBox.FReadSize(Index: integer): integer;
begin
  if (Index < Items.Count) then
    Result := GetData(Index)^.Size
  else
    Result := 0;
end;

function TXFileListBox.FReadAttribute(Index: integer): TXFileType;
begin
  if (Index >= 0) and (Index < Items.Count) then
    Result := AttrToFileType(GetData(Index)^.Attribute)
  else
    Result := [];
end;

function TXFileListBox.CompareFileName(const Index1, Index2: integer): integer;
begin
  Result := (CompareString(LOCALE_USER_DEFAULT, 0,
    PChar(FSortList.List^[Index1].FString), PData(FSortList.List^[Index1].FObject)^.FileNameLength,
    PChar(FSortList.List^[Index2].FString), PData(FSortList.List^[Index2].FObject)^.FileNameLength) - 2)
    * FSortOrderMultiplier;
end;

function TXFileListBox.CompareSize(const Index1, Index2: integer): integer;
begin
  Result := CompareInt(PData(FSortList.Objects[Index1])^.Size, PData(FSortList.Objects[Index2])^.Size) * FSortOrderMultiplier;
end;

function TXFileListBox.CompareDate(const Index1, Index2: integer): integer; // Date is integer part of TDateTime
begin
  Result := CompareInt(Round(PData(FSortList.Objects[Index1])^.Date), Round(PData(FSortList.Objects[Index2])^.Date)) * FSortOrderMultiplier;
end;

function TXFileListBox.CompareTime(const Index1, Index2: integer): integer; // time is fractional part ot TDateTime
begin
  Result := CompareInt(Round(Frac(PData(FSortList.Objects[Index1])^.Date)), Round(Frac(PData(FSortList.Objects[Index2])^.Date))) * FSortOrderMultiplier;
end;

function TXFileListBox.CompareAttribute(const Index1, Index2: integer): integer;
begin
  Result := CompareInt(PData(FSortList.Objects[Index1])^.Attribute, PData(FSortList.Objects[Index2])^.Attribute) * FSortOrderMultiplier;
end;

procedure TXFileListBox.ReSort;
begin
  Items.BeginUpdate;
  FSortList.Assign(Items);
  Sort;
  Items.Assign(FSortList);
  FSortList.Clear;
  Items.EndUpdate;
  Invalidate;
  Repaint;
end;

procedure TXFileListBox.Sort; // sorting data in FSortList
var
  Low: integer; // compute low measure of sorting
  Temp: integer; // temporary var for orig multiplier
begin
  FSortList.BeginUpdate;
  // sort folders
  if FFirstFolder <> -1 then
  begin
    Temp := FSortOrderMultiplier; // save original value
    case SortFolders of
      sfAlwaysByName:
        begin
          FSortOrderMultiplier := 1; // changed to be always ascending
          FSortList.CompareFunction := CompareFileName;
        end;
      sfLikeFiles: FSortList.CompareFunction := FCompareFunctions[SortBy];
    end;
    FSortList.QuickSort(FFirstFolder, FLastFolder); // sort only given range
    FSortOrderMultiplier := Temp;
  end;
  // sort files
  if (FLastFolder = -1) then
    if (FParentFolder = -1) then
      Low := 0
    else
      Low := 1
  else
    Low := FLastFolder + 1;
  if Low < (FSortList.Count - 1) then
  begin
    FSortList.CompareFunction := FCompareFunctions[SortBy];
    FSortList.QuickSort(Low, FSortList.Count - 1);
  end;
  FSortList.EndUpdate;
end;

procedure TXFileListBox.Update;
begin
  ReadFileNames;
end;

procedure TXFileListBox.CreateWnd;
begin
  inherited CreateWnd;
  ReadFileNames;
end;

function TXFileListBox.IsMaskStored: Boolean;
begin
  Result := DefaultMask <> FMask;
end;

function TXFileListBox.GetDrive: char;
begin
  Result := FDirectory[1];
end;

procedure TXFileListBox.Click;
begin
  inherited Click;
  if FLastSel <> ItemIndex then
    Change;
end;

procedure TXFileListBox.Change;
begin
  FLastSel := ItemIndex;
  if FFileEdit <> nil then
  begin
    if Length(GetFileName) = 0 then
      FileEdit.Text := Mask
    else
      FileEdit.Text := GetFileName;
    FileEdit.SelectAll;
  end;
  SetDirLabelCaption;
  if Assigned(FOnChange) then FOnChange(Self);
end;

procedure TXFileListBox.SetShowGlyphs(Value: Boolean);
begin
  if FShowGlyphs <> Value then
  begin
    FShowGlyphs := Value;
    if (FShowGlyphs = True) and (ItemHeight < (16 + 1)) then
      ResetItemHeight;
    AdjustWidth;
    Invalidate;
  end;
end;

procedure TXFileListBox.SetFileName(const NewFile: string);
begin
  if AnsiCompareFileName(NewFile, GetFileName) <> 0 then
  begin
    ItemIndex := SendMessage(Handle, LB_FindStringExact, 0,
      Longint(PChar(NewFile)));
    Change;
  end;
end;

procedure TXFileListBox.SetFileEdit(Value: TEdit);
begin
  FFileEdit := Value;
  if FFileEdit <> nil then
  begin
    FFileEdit.FreeNotification(Self);
    if GetFileName <> '' then
      FFileEdit.Text := GetFileName
    else
      FFileEdit.Text := Mask;
  end;
end;

procedure TXFileListBox.SetDrive(Value: char);
begin
  if (UpCase(Value) <> UpCase(FDirectory[1])) then
    ApplyFilePath(Format('%s:', [Value]));
end;

function TXFileListBox.SetDirectory(const NewDirectory: string): boolean;
begin
  Result := false;
  if AnsiCompareFileName(NewDirectory, FDirectory) <> 0 then
  begin
    { go to old directory first, in case not complete pathname
      and curdir changed - probably not necessary }
    if DirectoryExists(FDirectory) then
      ChDir(FDirectory);
//    ChDir(NewDirectory);
    Result := SetCurrentDir(NewDirectory);
    if Result then
    begin
      GetDir(0, FDirectory); { store correct directory name }
      ReadFileNames;
    end;
  end;
end;

procedure TXFileListBox.SetMask(const NewMask: string);
begin
  if FMask <> NewMask then
  begin
    FMask := NewMask;
    ReadFileNames;
  end;
end;

procedure TXFileListBox.CMFontChanged(var Message: TMessage);
begin
  inherited;
  ResetItemHeight;
end;

procedure TXFileListBox.ResetItemHeight;
var
  nuHeight: Integer;
begin
  nuHeight := GetItemHeight(Font);
  if (FShowGlyphs = True) and (nuHeight < (16 + 1)) then
    nuHeight := 16 + 1;
  ItemHeight := nuHeight;
end;

procedure TXFileListBox.ApplyFilePath(const EditText: string);
var
  DirPart: string;
  FilePart: string;
  NewDrive: Char;
begin
{$IFDEF DEBUG}
  if IsConsole then
    WriteLn(EditText);
{$ENDIF}
  if AnsiCompareFileName(FileName, EditText) = 0 then Exit;
  if Length(EditText) = 0 then Exit;
  if not ProcessPath(EditText, NewDrive, DirPart, FilePart) then Exit;
  if FDirList <> nil then
    FDirList.Directory := EditText
  else if NewDrive <> #0 then
    if not SetDirectory(Format('%s:%s', [NewDrive, DirPart])) then
      Exit
    else if not SetDirectory(DirPart) then
      Exit;
  if (Pos('*', FilePart) > 0) or (Pos('?', FilePart) > 0) then
    SetMask(FilePart)
  else if Length(FilePart) > 0 then
  begin
    SetFileName(FilePart);
    if FileExists(FilePart) then
    begin
      if GetFileName = '' then
      begin
        SetMask(FilePart);
        SetFileName(FilePart);
      end;
    end
    else
      raise EInvalidOperation.CreateFmt(rsInvalidFileName, [EditText]);
// make event FOnIvalidFileName ?
  end;
  if Assigned(FOnOpenFolder) then
    FOnOpenFolder(Directory);
end;

procedure TXFileListBox.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) then
  begin
    if (AComponent = FFileEdit) then
      FFileEdit := nil
    else if (AComponent = FDirList) then
      FDirList := nil
    else if (AComponent = FFilterCombo) then
      FFilterCombo := nil;
  end;
end;

{ TXFilterComboBox }

constructor TXFilterComboBox.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Style := csDropDownList;
  FFilter := rsDefaultFilter;
  MaskList := TStringList.Create;
end;

destructor TXFilterComboBox.Destroy;
begin
  MaskList.Free;
  inherited Destroy;
end;

procedure TXFilterComboBox.CreateWnd;
begin
  inherited CreateWnd;
  BuildList;
end;

function TXFilterComboBox.IsFilterStored: Boolean;
begin
  Result := rsDefaultFilter <> FFilter;
end;

procedure TXFilterComboBox.SetFilter(const NewFilter: string);
begin
  if AnsiCompareFileName(NewFilter, FFilter) <> 0 then
  begin
    FFilter := NewFilter;
    if HandleAllocated then BuildList;
    Change;
  end;
end;

procedure TXFilterComboBox.SetFileListBox(Value: TXFileListBox);
begin
  if FFileList <> nil then FFileList.FFilterCombo := nil;
  FFileList := Value;
  if FFileList <> nil then
  begin
    FFileList.FreeNotification(Self);
    FFileList.FFilterCombo := Self;
  end;
end;

procedure TXFilterComboBox.Click;
begin
  inherited Click;
  Change;
end;

function TXFilterComboBox.GetMask: string;
begin
  if ItemIndex < 0 then
    ItemIndex := Items.Count - 1;

  if ItemIndex >= 0 then
  begin
    Result := MaskList[ItemIndex];
  end
  else
    Result := '*.*';
end;

procedure TXFilterComboBox.BuildList;
var
  AFilter, MaskName, Mask: string;
  BarPos: Integer;
begin
  Clear;
  MaskList.Clear;
  AFilter := Filter;
  BarPos := AnsiPos('|', AFilter);
  while BarPos <> 0 do
  begin
    MaskName := Copy(AFilter, 1, BarPos - 1);
    Delete(AFilter, 1, BarPos);
    BarPos := AnsiPos('|', AFilter);
    if BarPos > 0 then
    begin
      Mask := Copy(AFilter, 1, BarPos - 1);
      Delete(AFilter, 1, BarPos);
    end
    else
    begin
      Mask := AFilter;
      AFilter := '';
    end;
    Items.Add(MaskName);
    MaskList.Add(Mask);
    BarPos := AnsiPos('|', AFilter);
  end;
  ItemIndex := 0;
end;

procedure TXFilterComboBox.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = FFileList) then
    FFileList := nil;
end;

procedure TXFilterComboBox.Change;
begin
  if FFileList <> nil then FFileList.Mask := Mask;
  inherited Change;
end;

{ TXPathLabel }

constructor TXPathLabel.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  AutoSize := False;
  ShowAccelChar := False;
end;

procedure TXPathLabel.Paint;
const
  Alignments: array[TAlignment] of Word = (DT_LEFT, DT_RIGHT, DT_CENTER);
  WordWraps: array[Boolean] of Word = (0, DT_WORDBREAK);
var
  Rect: TRect;
  Flags: Cardinal;
  S: string;
begin
  S := MinimizeName(Caption, Canvas, Width);  // instead of it use DT_END_ELLIPSIS or DT_PATH_ELLIPSIS with DrawText	
  with Canvas do
  begin
    if not Transparent then
    begin
      Brush.Color := Self.Color;
      Brush.Style := bsSolid;
      FillRect(ClientRect);
    end;
    Brush.Style := bsClear;
    Rect := ClientRect;
    Flags := DT_EXPANDTABS or DT_NOPREFIX or WordWraps[WordWrap] or Alignments[Alignment];
    { Calculate vertical layout }
    Canvas.Font := Self.Font;
    if not Enabled then
    begin
      OffsetRect(Rect, 1, 1);
      Canvas.Font.Color := clBtnHighlight;
      DrawText(Canvas.Handle, PChar(S), Length(S), Rect, Flags);
      OffsetRect(Rect, -1, -1);
      Canvas.Font.Color := clBtnShadow;
    end;
    DrawText(Canvas.Handle, PChar(S), Length(S), Rect, Flags);
  end;
end;

procedure Register;
begin
  RegisterComponents('Extra', [TXDriveComboBox, TXDirectoryListBox, TXFileListBox,
    TXFilterComboBox, TXPathLabel]);
end;

end.

